const Sequelize = require("sequelize");
const bcryptjs = require("bcryptjs");
const axios = require("axios");
const FormData = require("form-data");
const sequelizeConnect = require("../connection/db");

const Customers = sequelizeConnect.define(
  "customers",
  {
    id: {
      type: Sequelize.BIGINT,
      primaryKey: true,
      allowNull: false,
      autoIncrement: true,
    },
    type: {
      type: Sequelize.STRING(50),
      allowNull: false,
      defaultValue: "Buyer",
    },
    agent_code: {
      type: Sequelize.STRING(150),
      allowNull: true,
    },
    agent: {
      type: Sequelize.STRING(150),
      allowNull: true,
    },
    hardware_name: {
      type: Sequelize.STRING(150),
      allowNull: true,
    },
    first_name: {
      type: Sequelize.STRING(500),
      allowNull: false,
    },
    last_name: {
      type: Sequelize.STRING(500),
      allowNull: false,
    },
    email: {
      type: Sequelize.STRING(150),
      allowNull: false,
      unique: true,
      validate: {
        isEmail: true,
      },
    },
    password: {
      type: Sequelize.STRING(500),
      allowNull: false,
    },
    phone: {
      type: Sequelize.STRING(50),
      allowNull: true,
    },
    phone_2: {
      type: Sequelize.STRING(50),
      allowNull: true,
    },
    phone_3: {
      type: Sequelize.STRING(50),
      allowNull: true,
    },
    address: {
      type: Sequelize.STRING(500),
      allowNull: true,
    },
    longitude: {
      type: Sequelize.DOUBLE,
      allowNull: true,
    },
    latitude: {
      type: Sequelize.DOUBLE,
      allowNull: true,
    },
    buildcredit_beginning_balance: {
      type: Sequelize.DECIMAL(10, 2),
      allowNull: true,
    },
    buyer_type: {
      type: Sequelize.STRING(150),
      allowNull: true,
    },
    verificationToken: {
      type: Sequelize.STRING(500),
      allowNull: false,
    },
    verificationTokenTimestamp: {
      type: Sequelize.DATE,
      allowNull: false,
    },
    status: {
      type: Sequelize.ENUM("For Approval", "Verified"),
      allowNull: false,
      defaultValue: "For Approval",
    },
    created_timestamp: {
      type: Sequelize.DATE,
      allowNull: false,
      defaultValue: new Date(),
    },
    file_2303: {
      type: Sequelize.STRING(500),
      allowNull: true,
    },
    industry_classification: {
      type: Sequelize.STRING(200),
      allowNull: true,
    },
    account_status: {
      type: Sequelize.BOOLEAN,
      allowNull: false,
      defaultValue: true,
    },
    is_completed_profile: {
      type: Sequelize.BOOLEAN,
      allowNull: false,
      defaultValue: false,
    },
    account_source: {
      type: Sequelize.STRING(500),
      allowNull: true,
    },
    openauth_id: {
      type: Sequelize.STRING(500),
      allowNull: true,
    },
    client_facial_verification: {
      type: Sequelize.STRING(500),
      allowNull: true,
    },
    client_signature: {
      type: Sequelize.STRING(500),
      allowNull: true,
    },
  },
  {
    tableName: "customers",
    timestamps: false,
    hooks: {
      beforeCreate: async (customer) => {
        if (customer.changed("password")) {
          const saltRounds = 10;
          customer.password = await bcryptjs.hash(customer.password, saltRounds);
        }
      },
      beforeUpdate: async (customer) => {
        if (customer.changed("password")) {
          const saltRounds = 10;
          customer.password = await bcryptjs.hash(customer.password, saltRounds);
        }
      },
    },
  }
);

Customers.prototype.validatePassword = async function (plaintextPassword) {
  return bcryptjs.compare(plaintextPassword, this.password);
};
Customers.prototype._validatePasswordPhp = function (plaintextPassword) {
  const formData = new FormData();
  formData.append("password", plaintextPassword);

  return axios
    .post("https://buildhubware.com/generate-hash.php", formData)
    .then((response) => {
      const hashedPassword = response.data.hashedPassword;

      console.log(hashedPassword);
      console.log(this.password);
      console.log(response.data);
      return hashedPassword === this.password;
    })
    .catch((error) => {
      console.error("Error:", error);
      return false;
    });
};

module.exports = Customers;
