const SuggestedVoucher = require("../../models/SuggestedVouchers");
const UserCart = require("../../models/UserCart");
const Products = require("../../models/Products");
const { getUserCartExcludes, getProductUserCartExcludes } = require("../../helpers/model-excludes-attributes/excludes");
const SellersCompany = require("../../models/SellersCompany");
const Sequelize = require("sequelize");

exports.getAvailableVouchers = async (req, res, next) => {
  try {
    const { customerId } = req;

    // Get User Cart Seller Promise
    //#region
    const userCartPromise = UserCart.findOne({
      where: {
        user_id: customerId,
      },
      include: [
        {
          model: Products,
          as: "product",
          attributes: {
            exclude: getProductUserCartExcludes(),
          },
        },
      ],
      attributes: {
        exclude: getUserCartExcludes(),
      },
    });
    //#endregion

    // Get User Cart SUM
    //#region
    const totalCartPricePromise = UserCart.findOne({
      where: {
        user_id: customerId,
      },
      attributes: [[Sequelize.fn("SUM", Sequelize.col("product.price")), "total_cart_price"]],
      include: [
        {
          model: Products,
          as: "product",
          attributes: [],
        },
      ],
      raw: true,
    });
    //#endregion

    const [userCart, sumCart] = await Promise.all([userCartPromise, totalCartPricePromise]);

    // Validations
    //#region
    if (!userCart || !userCart.product || !userCart.product.seller_id) {
      return res
        .status(400)
        .json({ success: false, message: "There's a problem in getting seller information. Please try again later." });
    }
    if (!sumCart || !sumCart.total_cart_price) {
      return res
        .status(400)
        .json({ success: false, message: "There's a problem in getting cart information. Please try again later." });
    }
    //#endregion

    const sellerID = userCart.product.seller_id;
    const totalCartPrice = parseFloat(sumCart.total_cart_price);

    const suggestedVouchers = await SuggestedVoucher.findAll({
      where: {
        seller_company_id: sellerID,
        customer_id: customerId,
        status: true,
      },
    });

    if (!suggestedVouchers || !Array.isArray(suggestedVouchers) || suggestedVouchers.length === 0) {
      return res.status(200).json({ success: true, message: "No suggested vouchers for these cart items." });
    }

    const returnedResponse = [];
    const today = new Date();

    for (const voucher of suggestedVouchers) {
      const { id, description, min_amount_requirement, max_amount_requirement, voucher_amount, valid_until } = voucher;
      const validUntilDate = new Date(valid_until);

      if (
        totalCartPrice >= min_amount_requirement &&
        max_amount_requirement >= totalCartPrice &&
        today <= validUntilDate
      ) {
        returnedResponse.push({
          id,
          description,
          voucher_amount,
        });
      }
    }

    res.status(200).json({ success: true, vouchers: returnedResponse });
  } catch (err) {
    next(err);
  }
};

exports.suggestedVoucherHelper = async (req, suggestedVoucherId) => {
  try {
    const { customerId } = req;

    // Get User Cart Seller Promise
    //#region
    const userCartPromise = UserCart.findOne({
      where: {
        user_id: customerId,
      },
      include: [
        {
          model: Products,
          as: "product",
          attributes: {
            exclude: getProductUserCartExcludes(),
          },
        },
      ],
      attributes: {
        exclude: getUserCartExcludes(),
      },
    });
    //#endregion

    // Get User Cart SUM
    //#region
    const totalCartPricePromise = UserCart.findOne({
      where: {
        user_id: customerId,
      },
      attributes: [[Sequelize.fn("SUM", Sequelize.col("product.price")), "total_cart_price"]],
      include: [
        {
          model: Products,
          as: "product",
          attributes: [],
        },
      ],
      raw: true,
    });
    //#endregion

    const [userCart, sumCart] = await Promise.all([userCartPromise, totalCartPricePromise]);

    // Validations
    //#region
    if (!userCart || !userCart.product || !userCart.product.seller_id) {
      return { success: false, voucherAmount: 0 };
    }
    if (!sumCart || !sumCart.total_cart_price) {
      return { success: false, voucherAmount: 0 };
    }
    //#endregion

    const sellerID = userCart.product.seller_id;
    const totalCartPrice = parseFloat(sumCart.total_cart_price);

    const suggestedVoucher = await SuggestedVoucher.findOne({
      where: {
        id: suggestedVoucherId,
        seller_company_id: sellerID,
        customer_id: customerId,
        status: true,
      },
    });

    if (!suggestedVoucher) return { success: false, voucherAmount: 0 };

    const { id, min_amount_requirement, max_amount_requirement, voucher_amount, valid_until } = suggestedVoucher;
    const today = new Date();
    const validUntilDate = new Date(valid_until);

    if (
      totalCartPrice >= min_amount_requirement &&
      max_amount_requirement >= totalCartPrice &&
      today <= validUntilDate
    ) {
      return { success: true, voucherAmount: voucher_amount };
    } else {
      return { success: false, voucherAmount: 0 };
    }
  } catch (err) {
    return { success: false, voucherAmount: 0 };
  }
};
