const Sequelize = require("sequelize");
const Vouchers = require("../../models/Vouchers");

exports.getSellerVouchers = async (req, res, next) => {
  const { seller_id } = req;

  const { page = 1, limit = 20 } = req.pagination;
  const { keyword = "", amount_sort = "DESC" } = req.query;

  try {
    const whereClause = {
      seller_voucher: seller_id,
      status: 1,
    };

    const orderClause = [];

    if (amount_sort) orderClause.push(["amount", amount_sort]);

    if (keyword !== "" || keyword !== null) {
      const voucherWhere = {
        [Sequelize.Op.like]: `%${keyword}%`,
      };
      whereClause.voucher_code = voucherWhere;
    }

    const offset = (page - 1) * limit;

    const sellerVouchersPromise = Vouchers.findAll({
      where: whereClause,
      limit,
      offset,
      order: orderClause,
    });

    const sellerVoucherCountPromise = Vouchers.count({
      where: whereClause,
    });

    const [sellerVouchers, sellerVoucherCount] = await Promise.all([sellerVouchersPromise, sellerVoucherCountPromise]);

    let mappedSellerVoucher = [];
    if (sellerVouchers.length > 0) {
      mappedSellerVoucher = sellerVouchers.map((voucher) => ({
        id: voucher.id,
        voucher_code: voucher.voucher_code,
        amount: voucher.amount,
        amount_in_percent: voucher.amount_in_percent,
        voucher_amount_in_percent: voucher.in_percent === "Yes" ? true : false,
        valid_until: voucher.valid_until,
        status: voucher.status === 1 ? true : false,
      }));
    }

    const totalPages = Math.ceil(sellerVoucherCount / limit);
    const hasNextPage = (page - 1) * limit + sellerVouchers.length < totalPages;
    const hasPreviousPage = page > 1;

    res.status(200).json({
      success: true,
      vouchers: mappedSellerVoucher,
      pagination: {
        totalPages,
        currentPage: page,
        hasNextPage,
        hasPreviousPage,
      },
    });
  } catch (err) {
    next(err);
  }
};

exports.createSellerVoucher = async (req, res, next) => {
  const { seller_id } = req;
  try {
    const { voucher_code, amount, amount_in_percent, in_percent = false, valid_until } = req.body;

    if (!voucher_code || voucher_code === "" || voucher_code === null)
      return res.status(400).json({ success: false, message: "Invalid voucher code." });
    if (in_percent && !amount_in_percent)
      return res.status(400).json({ success: false, message: "Invalid amount in percent value." });
    if (!in_percent && !amount_in_percent && !amount)
      return res.status(400).json({ success: false, message: "Invalid voucher amount." });
    if (!valid_until) return res.status(400).json({ success: false, message: "Invalid voucher validity date." });

    const _voucherCheck = await Vouchers.findAll({
      where: {
        voucher_code,
      },
    });

    if (_voucherCheck.length > 0) return res.status(400).json({ success: false, message: "Voucher code exists!" });

    let voucherAmount = parseFloat(amount) || 0;
    if (in_percent) {
      const decimalValuePercent = parseFloat(amount_in_percent) / 100;
      voucherAmount = parseFloat(decimalValuePercent);
    }

    const createdVoucher = await Vouchers.create({
      voucher_code,
      amount: in_percent ? null : voucherAmount,
      amount_in_percent: in_percent ? voucherAmount : null,
      in_percent: in_percent ? "Yes" : null,
      valid_until,
      seller_voucher: seller_id,
      status: true,
      agent: "",
    });

    res.status(201).json({ success: true, createdVoucher });
  } catch (err) {
    next(err);
  }
};

exports.softDeleteVoucher = async (req, res, next) => {
  try {
    const { seller_id } = req;
    const { voucher_id } = req.params;

    const _voucher = await Vouchers.findOne({
      where: {
        id: voucher_id,
        seller_voucher: seller_id,
      },
    });

    if (!_voucher) return res.status(400).json({ success: false, message: "Voucher does not exists!" });

    _voucher.status = false;
    await _voucher.save();

    res.status(200).json({ success: true });
  } catch (err) {
    next(err);
  }
};
