const ProductRating = require("../models/ProductRating");
const Customers = require("../models/Customers");
const ProductMenu = require("../models/ProductMenu");
const Products = require("../models/Products");
const DetailProductReviews = require("../models/DetailProductReviews");
const errorHandler = require("../util/errorHandler");

exports.addProductRating = (req, res, next) => {
    const { customerId } = req;
    const { referenceNo } = req.params;

    const {
        productId,
        orderFormId,
        productRate,
        productReview,
    } = req.body;

    if(!orderFormId) {
        return errorHandler("Reference Number is missing from the request!", 400);
    }

    ProductRating.findOne({
        where: {
            customer_id: customerId,
            reference_no: referenceNo,
        },
    })
    .then((rate) => {
        if(!rate) {
            return errorHandler("Reference Number does not exists!", 401);
        }

        return new ProductRating({
            reference_no: referenceNo,
            order_form_id: orderFormId,
            customer_id: customerId,
            product_id: productId,
            rate: productRate,
            reviews: productReview,
        }).save();
    })
    .then(() => {
        res.status(201).json({ success: true });
    })
    .catch((err) => {
        next(err);
    });
};
exports.getAllProductRating = async (req, res, next) => {
    try {
        const reviews = await DetailProductReviews.findAll({
            include: [
                {
                    model: Customers,
                    as: 'customer',
                    attributes: ["id", "first_name", "last_name", "email", "phone", "address"]
                },
                {
                    model: Products,
                    as: 'product',
                    attributes: ["id", "menu", "photo"]
                }
            ]
        });

        const reviewResponse = reviews.map(review => ({
            id: review.id,
            timestamp: review.timestamp,
            customer: {
                customer_id: review.customer.id,
                customer_first_name: review.customer.first_name,
                customer_last_name: review.customer.last_name,
                email: review.customer.email,
                phone: review.customer.phone,
                address: review.customer.address,
                rate: review.rate,
                reviews: review.reviews
            },
            product: {
                product_id: review.product.id,
                product_menu: review.product.menu,
                photo_url: review.product.photo ? `https://buildhub.ph/img/products/${review.product.photo}` : null
            }
        }));

        res.status(200).json(reviewResponse);
    } catch (error) {
        next(error);
    }
};

exports.getProductRatingId = async (req, res, next) => {
    try {
        const productId = req.params.productId;

        const reviews = await DetailProductReviews.findAll({
            where: { product_id: productId },
            include: [
                {
                    model: Customers,
                    as: 'customer',
                    attributes: ["id", "first_name", "last_name", "email", "phone", "address"]
                },
                {
                    model: Products,
                    as: 'product',
                    attributes: ["id", "menu", "photo"]
                }
            ]
        });

        if (!reviews || reviews.length === 0) {
            return res.status(404).json({ success: false, message: "Product is not yet rated" });
        }

        const reviewResponse = reviews.map(review => ({
            id: review.id,
            timestamp: review.timestamp,
            customer: {
                customer_id: review.customer.id,
                customer_first_name: review.customer.first_name,
                customer_last_name: review.customer.last_name,
                email: review.customer.email,
                phone: review.customer.phone,
                address: review.customer.address,
                rate: review.rate,
                reviews: review.reviews
            },
            product: {
                product_id: review.product.id,
                product_menu: review.product.menu,
                photo_url: review.product.photo ? `https://buildhub.ph/img/products/${review.product.photo}` : null
            }
        }));

        res.status(200).json(reviewResponse);
    } catch (error) {
        next(error);
    }
};
